#include <cstdlib>
#include <iostream>
#include <vector>
#include <thread>
#include <mutex>
#include <condition_variable>

using namespace std;

// Inspired from https://riptutorial.com/cplusplus/example/30142/semaphore-cplusplus-11
class Semaphore {
public:
  Semaphore (int count_ = 0) : count(count_) {}
  
  inline void acquire() {
    unique_lock<mutex> lock(mtx);
    while(count == 0) cv.wait(lock); // When no tokens are free, the thread is put on hold.
    count--; // Token consumption
  }
  
  inline void release() {
    unique_lock<mutex> lock(mtx);
    count++;
    cv.notify_one(); // notify the waiting thread
  }
  
private:
  mutex mtx;
  condition_variable cv;
  int count;
};

// inspired from https://www.mkyong.com/java/java-thread-mutex-and-semaphore-example/
void task(int name, Semaphore *semaphore) {
  cout << "Thread " << name << " acquiring lock..." << endl;
  semaphore->acquire();
  cout << "Thread " << name << " got the permit!" << endl;
  for( int i = 0 ; i < 5 ; ++i ) {
    cout << "Thread " << name << " is performing operation " << i << endl;
    // sleep 1 second
    this_thread::sleep_for(chrono::milliseconds(1000));
  }
  cout << "Thread " << name << " releasing lock..." << endl;
  semaphore->release();
}

// Main program
int main(int argc, char** argv) {
  int argument;
  if(argc<2) { cout << "argument? "; cin >> argument; } else { argument=atoi(argv[1]); }

  // Create a semaphore
  Semaphore semaphore(argument);

  // Start threads
  vector<thread> threads(6);
  for(int i = 0; i < 6; i++){
    threads[i] = thread(task, i, &semaphore);
  }

  // End of the program
  for(int i = 0; i < 6; i++){
    threads[i].join();
  }
  return EXIT_SUCCESS;
}

